
# imagetiler.py:  Processes images for storage and access on the PNW Herbaria Web Server
#
# Ben Legler
# 6/14/2010


# Windows command-line usage:
# C:\Python27\python.exe -W ignore::DeprecationWarning C:\PNWHerbaria\Scripts\imagetiler.py


# This script processes any JPEG images it finds in the "JPEG" sub-folder of imageDirs['dropboxDir'].
# For each JPEG, it does the following steps:
#  1) Opens the JPEG and rotates it as specified in the herbaria[] array settings below.  Also remove 
#     any underscore ("_") present in the image name between the acronym and image number.
#  2) Creates tiles for use in the online image viewer, sending these to imageDirs['tileDir'].
#  3) Moves the JPEG to an archival folder imageDirs['jpegDir'].
#  4) Check for a corresponding CR2 image and move it to the archival folder imageDirs['cr2Dir'], optionally
#     converting to DNG (Digital Negative) in the process.
#  5) Add an entry for this image to the Images table in the database specified in the herbaria[] array, 
#     Extracting EXIF date/time from the CR2 image in the process (to later link image to folder).

# This script assumes images are captured in RAW format (.CR2), with JPEGs subsequently created 
# using Canon's batch conversion software.  The CR2 and JPEgs should both be accessible by this 
# script (the JPEG is used to creat tiles; the CR2 to archive, possibly as DNG).  The reason 
# for this is because ImageMagick/dcraw does a poor job of converting CR2 to JPEG or TIFF, so 
# we can't automate the conversion within this script and still maintain high image quality.
#
# The tile files are named according to the image name.  These are placed in a folder of the 
# same name that is itself placed in a folder called "Tiles" with subfolders for each acronym 
# and for ranges of image numbers (the acronym subfolders and image number subfolders names 
# are determined from the image name itself (e.g., "MONT_014849.CR2" -> "Tiles/MONT/10000/MONT014849/").
#
# Individual tiles are named according to the template z_c_r.jpg where z = zoom level, 
# c = column, r = row. Tiles are stored into a single physical file that is a simple 
# concatenation of all the tile jpegs. In addition, a text file is generated that contains 
# image metadata and a list of tile byte positions within the combined file.
#
# The JPEG and RAW images are stored in corrsponding archive folders, with subfolders created 
# for the acronyms and ranges of image numbers in the same manner as described for the tiles.
#
# An extraction script (e.g., tile.php) is needed to pull individual tiles from the combined 
# file when requested by the image viewer.
# 
# Multiple copies of this script can be run simultaneously, thereby taking advantage of the 
# server's quad-cores and greatly reducing tiling times.  To do so, split the Dropbox JPEG and 
# CR2 images into several subfolders, and set each copy of the script to run on each pair of 
# these subfolders.  A good method of splitting is by state or collection.
#
#
# Requirements:
#  1) Python (tested with version 2.6.4 and 2.7.1)
#     (http://www.python.org/download/)
#  2) Python Image Library (tested with version 1.1.6)
#     (http://www.pythonware.com/products/pil/)
#  3) Adobe DNG Converter, for converting .CR2 images to .DNG for archiving (optional).
#     (Note: "Adobe DNG Converter.exe" must be renamed to remove the spaces, and moved to a path that has no spaces)
#     (http://www.adobe.com/products/dng/)
#  4) MySQL-Python, for creating blank records in the database (tested with version 1.2.2.win32-py2.6).
#     (Windows builds: http://www.codegood.com/archives/4)
#     (http://mysql-python.sourceforge.net/)
#  5) ExifTool, for extracting date taken from the RAW images EXIF data for insertion into MySQL database.
#     (http://www.sno.phy.queensu.ca/~phil/exiftool/)


# CONFIGURATION:

# New line character for metadata text file output. Change this to suite your web server's OS ("\r", "\n", or "\r\n"):
newLine = "\r\n"

# Path to the "Adobe DNG Converter" executable:
dngConverterPath = 'C:\\PNWHerbaria\\Scripts\\Adobe_DNG_Converter.exe'

# Flag indicating whether to convert CR2 images to DNG for archiving:
convertToDNG = False

# Path to exiftool executable (used to extract date taken from RAW image):
exiftoolPath = "C:\\PNWHerbaria\\Scripts\\exiftool.exe"

# Location in which to store the log file output by this script:
logDir = 'C:\\PNWHerbaria\\Scripts\\TilerLogs\\'

# List of file types that will be opened with PIL for tiling:
# PIL can handle JPEG and 8-bit TIFF images.  To process other image types it may be
# necessary to modify this script to use ImageMagick instead of PIL.
# This list is case-insensitive.
fileTypes = '\.(jpg|jpeg|tif|tiff)$'

# Image directories:
imageDirs = {
'dropboxDir': 'C:\\PNWHerbaria\\Dropbox\\',    # Location of the unprocessed RAW and JPEG images (in sub-folders)
'rawDir': 'C:\\PNWHerbaria\\Images\\RAW\\',    # Directory in which to store the archival RAW images (CR2 or DNG)
'jpegDir': 'C:\\PNWHerbaria\\Images\\JPEG\\',  # Directory in which to store the JPEG images
'tileDir': 'C:\\PNWHerbaria\\Images\\Tiles\\'  # Directory in which to store the tiles for the image viewer
}

# Settings for tiles created for the image viewer:
# (see also 'ImageScale' and 'ImageScaleUnit' under individual herbaria settings below)
tileSettings = {
'tileSize': 256,          # Width and height of tiles, in pixels
'thumbnailWidth': 150,    # Width of thumbnail image, in pixels (height will scale proportionately)
}

# Array to hold database and image metadata settings for each herbarium or collection:
# Add blocks as needed.
herbaria = {
'WWB': {
'acronym': 'WWB',        # Acronym of this herbarium/collection (must be identical to the acronym used for images)
'addToDB': True,         # Flag indicating whether images should be linked to the database for this collection, if one exists.
'MySQLServer': '',       # Database connection (URL or IP address)
'MySQLUser': '',         # Database user name
'MySQLPassword': '',     # Database password
'MySQLDatabase': '',     # Database name
'copyright': '',         # This is inserted into the image medadata text file used in the image viewer
'copyrightURL': '',      # This is inserted into the image medadata text file used in the image viewer
'rotate': 90,            # Number of degrees by which to rotate the image clockwise (enter 0 for no rotation)
'imageScale': 320,       # Scale, in pixels per unit
'imageScaleUnit': 'in'   # Scale unit, can be one of: "mm", "cm", "m", "km", "in", "ft", "mi" (or define others as desired)
}
}

# END CONFIGURATION


import os, os.path, sys, re, shutil, _mysql
import time
from time import strftime
from math import *
from PIL import Image
import ImageEnhance
import warnings


# Accepts a source image and creates tiles
class Tiler:

    def __init__(self, jpegPath, name, baseName, tilePath, acronym):
        # Reads in the source image and sets some image parameters:
        
        self.jpegPath = jpegPath
        self.name = name
        self.baseName = baseName
        self.tilePath = tilePath
        self.acronym = acronym
        self.copyright = herbaria[self.acronym]['copyright']
        self.copyrightURL = herbaria[self.acronym]['copyrightURL']
        self.validImage = False
        print "Opening", self.name,
        
        # Directly open the image with PIL (this works with JPEGs and some TIFFs; for other images ImageMagick may be required):
        try:
            self.sourceImage = Image.open(self.jpegPath)
            self.validImage = True
            # Optionally rotate image:
            if herbaria[self.acronym]['rotate'] != 0:
                print "\nRotating %i degrees clockwise" % (herbaria[self.acronym]['rotate']),
                self.sourceImage = self.sourceImage.rotate(-herbaria[self.acronym]['rotate']) # convert to counter-clockwise for PIL
                self.sourceImage.save(self.jpegPath, "jpeg", quality=99)  # 99 is just right to give ca. 7 MB files; 98 is too small and 100 is too large
        except:
            # Add ImageMagick code here if necessary
            print "COULD NOT OPEN SOURCE JPEG!!!  TILES NOT CREATED!"
            self.validImage = False
        
        if self.validImage == True:
            self.imageWidth = self.sourceImage.size[0]
            self.imageHeight = self.sourceImage.size[1]
            
            # calculate number of zoom levels (zero-based):
            widthMin = (log(self.imageWidth) - log(tileSettings['tileSize'])) / log(2)
            heightMin = (log(self.imageHeight) - log(tileSettings['tileSize'])) / log(2)
            self.layerCount = int(ceil(max(widthMin, heightMin)))
         
    def generateTiles(self):
        # Tile the image, starting at the max zoom level and working out until the entire image fits into a single tile:
        # Also create the metadata file as the tiles are generated.
        # Also create the thumbnail one step before the image width drops below the thumbnail width.
        
        if self.validImage == False:
            return
        
        if not os.path.exists(self.tilePath):
            os.makedirs(self.tilePath)
        
        # Remove the tileFile if it already exists (it will be opened for appending, so if it exists then the new tiles will be improperly appended to the old tiles)
        if os.path.exists(os.path.join(self.tilePath, self.baseName + ".tls")):
            os.remove(os.path.join(self.tilePath, self.baseName + ".tls"))

        # Create/open the metadata file and tile file:
        metaFile = open(os.path.join(self.tilePath, self.baseName + ".txt"), "wb")  # write, binary mode
        tileFile = open(os.path.join(self.tilePath, self.baseName + ".tls"), "ab")  # append, binary mode
        
        print "\nGenerating metadata file: %s.txt" % (self.baseName)
        
        # Output metadata tags:
        metaFile.write("name\t%s%s" % (self.baseName, newLine))
        metaFile.write("copyright\t%s%s" % (self.copyright, newLine))
        metaFile.write("copyrighturl\t%s%s" % (self.copyrightURL, newLine))
        metaFile.write("width\t%i%s" % (self.imageWidth, newLine))
        metaFile.write("height\t%i%s" % (self.imageHeight, newLine))
        metaFile.write("maxzoom\t%i%s" % (self.layerCount, newLine))
        metaFile.write("tilesize\t%i%s" % (tileSettings['tileSize'], newLine))
        metaFile.write("tileoverlap\t0%s" % (newLine))
        metaFile.write("scale\t%i%s" % (herbaria[self.acronym]['imageScale'], newLine))
        metaFile.write("scaleunit\t%s%s" % (herbaria[self.acronym]['imageScaleUnit'], newLine))
        metaFile.write("overlays\t[]%s" % (newLine))
        metaFile.write("format\tjpg%s" % (newLine))

        # Loop through each zoom layer, starting with the max zoom and stepping down to the min zoom, adding each tile to tileFile:
        # Tiles are ordered in tileFile from max zoom to min zoom by col then by row.
        currentLayer = self.layerCount
        tileStart = 0
        tileLength = 0
        print "Generating tiles:",
        while currentLayer >= 0:
            print "%s," % currentLayer,
            rows = int(ceil(self.imageHeight / tileSettings['tileSize']))
            cols = int(ceil(self.imageWidth / tileSettings['tileSize']))
            
            # Loop through rows and cols for this zoom level, and create tiles:
            for c in range(0, cols+1):
                for r in range(0, rows+1):
                    tileName = "%i_%i_%i.jpg" % (currentLayer, c, r)
                    if (self.imageWidth - c*tileSettings['tileSize']) > 0 and (self.imageHeight - r*tileSettings['tileSize']) > 0:
                        tileLeft = c*tileSettings['tileSize']
                        tileTop = r*tileSettings['tileSize']
                        tileRight = min(c*tileSettings['tileSize'] + tileSettings['tileSize'], self.imageWidth)
                        tileBottom = min(r*tileSettings['tileSize'] + tileSettings['tileSize'], self.imageHeight)
                        tile = self.sourceImage.crop((tileLeft, tileTop, tileRight, tileBottom))
                        if tile.size[0] < tileSettings['tileSize'] or tile.size[1] < tileSettings['tileSize']:
                            tile2 = Image.new("RGB", (256,256), 0x000000)
                            tile2.paste(tile, (0,0))
                            tile2.save(tileFile, "jpeg", quality=90)
                        else:
                            tile.save(tileFile, "jpeg", quality=90)
                        tileFile.flush()
                        tileLength = tileFile.tell() - tileStart
                        metaFile.write("%s\t%i\t%i%s" % (tileName, tileStart, tileLength, newLine))
                        tileStart = tileFile.tell()
            
            # Reduce the image size by 1/2 for the next zoom level:
            currentLayer = currentLayer - 1
            self.imageWidth = int(ceil(self.imageWidth / 2))
            self.imageHeight = int(ceil(self.imageHeight / 2))
            self.sourceImage = self.sourceImage.resize((self.imageWidth, self.imageHeight), Image.ANTIALIAS)
            
            # Create the thumbnail image:
            thumbnailWidth = tileSettings['thumbnailWidth']
            if self.imageWidth > thumbnailWidth and (self.imageWidth/2) <= thumbnailWidth:
                print "thumbnail,",
                thumbnailHeight = int(ceil(float(self.imageHeight) * (float(thumbnailWidth) / float(self.imageWidth))))
                thumbnail = self.sourceImage.resize((thumbnailWidth, thumbnailHeight), Image.ANTIALIAS)
                thumbnail.save(os.path.join(self.tilePath, self.baseName + "_tb.jpg"), "jpeg", quality=85)
        
        metaFile.close()
        tileFile.close()


# Connects to a MySQL database and inserts/updates specimen records and image records as needed:
class SpecimenDatabase:
    
    def __init__(self, acronym):
        # Connects to MySQL:
        self.acronym = acronym
        h = herbaria[self.acronym]
        self.db = _mysql.connect(h['MySQLServer'], h['MySQLUser'], h['MySQLPassword'], h['MySQLDatabase'])
        
    def addEntry(self, imageName, dateImaged, timeImaged):
        # Update the MySQL database for this image:
        
        h = herbaria[self.acronym]
        
        acronymAndNumber = re.sub('[_\-]{1}[A-Za-z0-9]{1}$', '', imageName)  # remove suffix if any
        acronym = re.sub('[_\-0-9]+.*', '', acronymAndNumber)  # remove everything after the first dash or underscore or digit
        
        imageID = 0
        specimenID = 0
        imagedBy = ""
        
        # See if this image already exists in the images table:
        self.db.query("SELECT ID FROM images WHERE ImageName=\"%s\"" % (imageName))
        r = self.db.store_result()
        if r.num_rows() > 0:
            row = r.fetch_row()
            if row != None:
                if row[0]:
                    if row[0][0] != None:
                        imageID = int(row[0][0])
        
        # See if this image is already linked to a specimen record:
        self.db.query("SELECT SpecimenID FROM images WHERE ImageName LIKE\"%s%s\"" % (acronymAndNumber, '%'))
        r = self.db.store_result()
        if r.num_rows() > 0:
            row = r.fetch_row()
            if row != None:
                if row[0]:
                    if row[0][0] != None:
                        specimenID = int(row[0][0])
        
        # If not, try to match the image name to the barcode of an existing specimen record, and link to that record:
        if specimenID < 1:
            self.db.query("SELECT ID FROM specimens WHERE Barcode=\"%s\"" % (acronymAndNumber))
            r = self.db.store_result()
            if r.num_rows() > 0:
                row = r.fetch_row()
                if row != None:
                    if row[0]:
                        if row[0][0] != None:
                            specimenID = int(row[0][0])
        
        # Try to match this image to a folder metadata record:
        folderID = 'NULL'
        imagedBy = ''
        family = ''
        folderName = ''
        folderCode = ''
        self.db.query("SELECT ID, TimeImaged, Family, FolderName, FolderCode, ImagedBy FROM folders WHERE DateImaged=\"%s\" ORDER BY TimeImaged DESC" % (dateImaged))
        r = self.db.store_result()
        if r.num_rows() > 0:
            row = r.fetch_row()
            while (row):
               if row[0][1] != "" and row[0][1] < timeImaged:
                   if row[0][0] != None:
                       folderID = str(row[0][0])
                       family = str(row[0][2])
                       folderName = str(row[0][3])
                       folderCode = str(row[0][4])
                       imagedBy = str(row[0][5])
                       break
               row = r.fetch_row()
        
        # Add or update this image in the images table:
        if imageID < 1 and specimenID < 1:
            # New image, doesn't match to an existing specimen record:
            print "Adding %s to images table in %s's MySQL database." % (imageName, self.acronym)
            self.db.query("INSERT INTO images (AllowOnline, SpecimenID, ImageName, ImagedBy, DateImaged, TimeImaged, Family, FolderName, FolderCode, Notes) VALUES ('Y', NULL, \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", NULL)" % (imageName, imagedBy, dateImaged, timeImaged, family, folderName, folderCode))
        elif imageID < 1 and specimenID > 0:
            # New image, matches to an existing specimen record:
            print "Adding %s to images table in %s's MySQL database." % (imageName, self.acronym)
            self.db.query("INSERT INTO images (AllowOnline, SpecimenID, ImageName, ImagedBy, DateImaged, TimeImaged, Family, FolderName, FolderCode, Notes) VALUES ('Y', %i, \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", \"%s\", NULL)" % (specimenID, imageName, imagedBy, dateImaged, timeImaged, family, folderName, folderCode))
        elif imageID > 0 and specimenID < 1:
            # Existing image, doesn't match to an existing specimen record:
            print "Updating entry for %s in images table in %s's MySQL database." % (imageName, self.acronym)
            self.db.query("UPDATE images SET SpecimenID=NULL, Imagedby=\"%s\", DateImaged=\"%s\", TimeImaged=\"%s\", Family=\"%s\", FolderName=\"%s\", FolderCode=\"%s\" WHERE ImageName=\"%s\"" % (imagedBy, dateImaged, timeImaged, family, folderName, folderCode, imageName))
        elif imageID > 0 and specimenID > 0:
            # Existing image, matches to an existing specimen record:
            print "Updating entry for %s in images table in %s's MySQL database." % (imageName, self.acronym)
            self.db.query("UPDATE images SET SpecimenID=%i, Imagedby=\"%s\", DateImaged=\"%s\", TimeImaged=\"%s\", Family=\"%s\", FolderName=\"%s\", FolderCode=\"%s\" WHERE ImageName=\"%s\"" % (specimenID, imagedBy, dateImaged, timeImaged, family, folderName, folderCode, imageName))


if __name__ == "__main__":
    
    # Open/create a simple log file to record tiler results:
    logfile = open(os.path.join(logDir, 'tilerlog-ID-_' + strftime("%Y%m%d_%H%M%S") + ".txt"), 'w')
    
    print "\n-------------------------------------------"
    print "Batch tiler run started on", strftime("%Y-%m-%d %H:%M:%S")
    print "Dropbox folder: %s" % imageDirs['dropboxDir']
    
    print >> logfile, "\nBatch tiler run started on", strftime("%Y-%m-%d %H:%M:%S"),"\n"
    print >> logfile, "Dropbox folder: %s" % imageDirs['dropboxDir']
    
    acronymNum = 0
    processed = 0
    errors = 0
    lastAcronym = ''
    
    # Loop through image files in this dropbox folder (including subfolders, if any):
    for root, dirs, files in os.walk(os.path.join(imageDirs['dropboxDir'], 'JPEG')):
        if len(files) < 1:
            print "\nThere are no JPEG images to process"
            print >> logfile, "\nThere are no JPEG images to process"
        for name in files:
            
            # Only process files if they are one of the specified image formats and do not begin with "._" (Mac junk):
            if re.search(fileTypes, name, re.IGNORECASE) and name.startswith('._') == False:
                
                print "-------------------------------------------"
                
                # Determine the final image name and path settings:
                jpegPath = os.path.join(root, name)
                baseName = re.sub('\.\w+$', '', name)
                newName = baseName
                if re.search('^[a-zA-Z]+_', baseName):
                    if re.search('^[a-zA-Z]+_', baseName).group(0) != "":
                        newName = re.sub('_', '', baseName, 1)
                acronym = re.search('^[a-zA-Z]+', baseName).group(0)
                imageNumber = re.search('[0-9]+', baseName).group(0)
                roundedNumber = str(int(floor(int(imageNumber)/10000) * 10000))
                tilePath = os.path.join(imageDirs['tileDir'], acronym, roundedNumber, newName)
                
                # See if a tiled image with the same name already exists; if so, log a message (but still tile the image and overwrite the existing one):
                tileExists = os.path.exists(tilePath)
                if tileExists:
                    print "TILES ALREADY EXIST AND WERE OVERWRITTEN FOR", name
                    print >> logfile, "TILES ALREADY EXIST AND WERE OVERWRITTEN FOR", name
                else:
                    os.makedirs(tilePath)
                
                # Read in the JPEG and initialize image parameters:
                # JPEG is also rotated at this point if necessary.
                tiledImage = Tiler(jpegPath, name, newName, tilePath, acronym)
                
                # Generate metadata file, tiles, and thumbnail image:
                if tiledImage.validImage:
                    print >> logfile, "Tiling", jpegPath
                    tiledImage.generateTiles()
                else:
                    print >> logfile, "ERROR OPENING IMAGE: ", jpegPath
                    
                # Check to see that the tiled image actually exists. If not, log an error message and do nothing further:
                if os.path.exists(os.path.join(tilePath, newName + ".tls")) == False:
                    print "ERROR: FAILURE CREATING TILES FOR", name
                    print >> logfile, "ERROR: FAILURE CREATING TILES FOR", name
                    errors = errors + 1
                else:
                    dropboxRawPath = os.path.join(imageDirs['dropboxDir'], 'RAW', baseName + ".CR2")
                    archiveRawDir = os.path.join(imageDirs['rawDir'], acronym, roundedNumber)
                    archiveRawPath = os.path.join(archiveRawDir, newName + ".CR2")
                    
                    # Extract "Date Taken" from EXIF data in the RAW image for insertion into images table in database:
                    if herbaria[acronym]['addToDB'] == True:
                        print "\nExtracting date taken from CR2 image EXIF data",
                        dateImaged = '2010-06-01'  # default value if EXIF read fails
                        timeImaged = '00:00:00'    # default value if EXIF read fails
                        os.system("%s -createdate -w .txt %s" % (exiftoolPath, dropboxRawPath)) # Extract only the date taken tag
                        exifTextPath = os.path.join(imageDirs['dropboxDir'], 'RAW', baseName + ".txt")
                        exifFile = open(exifTextPath, 'r')
                        exifLine = exifFile.readline()
                        exifFile.close()
                        exifParts = exifLine.split(': ')
                        if exifParts[1] != "":
                            dateParts = exifParts[1].split()
                            dateImaged = dateParts[0].replace(":", "-")
                            timeImaged = dateParts[1]
                        if os.path.exists(exifTextPath):
                            os.remove(exifTextPath)
                    
                    # Add an entry for this image to the MySQL database for this collection:
                    if herbaria[acronym]['addToDB'] == True and herbaria[acronym]['acronym'] != '':
                        if acronym != lastAcronym:
                            dbConnect = SpecimenDatabase(acronym)
                        dbConnect.addEntry(newName, dateImaged, timeImaged)
                    
                    # Move JPEG image to the JPEG archive (also renaming if necessary to remove underscore):
                    print "Moving JPEG image to JPEG archive"
                    dropboxJpegPath = os.path.join(imageDirs['dropboxDir'], 'JPEG', baseName + ".jpg")
                    archiveJpegDir = os.path.join(imageDirs['jpegDir'], acronym, roundedNumber)
                    archiveJpegPath = os.path.join(archiveJpegDir, newName + ".jpg")
                    if not os.path.exists(archiveJpegDir):
                        os.makedirs(archiveJpegDir)
                    # If the jpeg already exists, then deleted it or else the os.rename function will throw an error:
                    if os.path.exists(archiveJpegPath):
                        os.remove(archiveJpegPath)
                    if os.path.exists(dropboxJpegPath):
                        os.rename(dropboxJpegPath, archiveJpegPath)
                    
                    # Move RAW image, if present, to RAW archive (also renaming, and optionally converting to DNG in the process):
                    if os.path.exists(dropboxRawPath):
                        if not os.path.exists(archiveRawDir):
                            os.makedirs(archiveRawDir)
                        if convertToDNG:
                            print "Converting CR2 to DNG and moving to RAW archive"
                            # If the DNG already exists then remove it (otherwise the Adobe DNG Converter will create a second copy with an "_1" appended)
                            if os.path.exists(os.path.join(archiveRawDir, newName + ".dng")):
                                os.remove(os.path.join(archiveRawDir, newName + ".dng"))
                            if os.path.exists(dropboxRawPath):
                                os.system("%s -d %s -o %s %s" % (dngConverterPath, archiveRawDir, newName + ".dng", dropboxRawPath))
                                os.remove(dropboxRawPath)
                        else:
                            print "Moving CR2 image to RAW archive"
                            # If the archived CR2 already exists, then deleted it or else the os.rename function will throw an error:
                            if os.path.exists(archiveRawPath):
                                os.remove(archiveRawPath)
                            if os.path.exists(dropboxRawPath):
                                os.rename(dropboxRawPath, archiveRawPath)
                    
                    processed = processed + 1
                    
            lastAcronym = acronym
            acronymNum = acronymNum + 1
            
    print "-------------------------------------------"
    print "Batch tiler run completed on %s. %i images processed. %i errors encountered." % (strftime("%Y-%m-%d %H:%M:%S"), processed, errors)
    print >> logfile, "\nBatch tiler run completed on %s. %i images processed. %i errors encountered." % (strftime("%Y-%m-%d %H:%M:%S"), processed, errors)
    
    logfile.close()
